function [rmse, cds_new] = RMSE_CDS_mispricing(cds, start_date_num, end_date_num, maturities, relative)
% --------------------------------------------------------------------------------------------------
% Calculate (time series of) square root of average squared CDS pricing error between 'start_date' and
% 'end_date'. Note that model-implied CDS spreads need to be calculated beforehand. This function
% simply adds up all squared pricing errors.
% --------------------------------------------------------------------------------------------------
% cds               ... credit default swap structure (see 'all_steps_in_a_row.m')
% start_date_num    ... datenum of start date
% end_date_num      ... datenum of end date
% maturities        ... indicator vector which maturities to include
% relative          ... whether to calculate relative (default) or absolute pricing errors
% --------------------------------------------------------------------------------------------------
% sample call: RMSE_CDS_mispricing(CDX_NA_IG.portfolio(1), datenum('09/20/2005'), datenum('03/17/2006'), [1 0 0])
% --------------------------------------------------------------------------------------------------

% Set default values
if (nargin <= 1)
    start_date_num = cds.dates{1}(1);
    end_date_num = cds.dates{1}(end);
end
if (nargin <= 2)
    maturities = ones(length(cds.dates), 1);
end
if (nargin <= 4)
    relative = 1;
end

% Sum up squared pricing erros for all desired maturities
for i=1:length(maturities)
    if (length(cds.model_price) < i)
        continue;
    end
    if (maturities(i) == 1)    
        if (isempty(cds.model_price{i}))
            continue;
        end
        
        % Determine relevant date range
        start_pos = find(cds.dates{i} >= start_date_num, 1, 'first');
        end_pos = find(cds.dates{i} <= end_date_num, 1, 'last');
        used_range = start_pos:end_pos;
        
        % Allocate memory for output
        if (i==1)
            date_range = cds.dates{1}(used_range);
            num_dates  = end_pos - start_pos + 1;
            num_prices = zeros(num_dates, 1);
            errors = zeros(num_dates, 1);
            used_range_5yr = used_range;
        end
        
        % Extract model_implied and market prices of tranches
        market_price = cds.market_price{i}(used_range,:);
        model_price = cds.model_price{i}(used_range,:);
        
        % Determine positions where pricing errors match
        [trash, ai, bi] = intersect_sorted(date_range, cds.dates{i}(used_range));
        if (isempty(ai))
            continue;
        end
        
        % Add squared pricing errors
        num_prices(ai) = num_prices(ai) + size(market_price, 2);
        if (relative == 1)
            rel_error_price = ((market_price - model_price)./market_price).^2;
            rel_error_price = sum(rel_error_price,2);
            errors(ai) = errors(ai) + rel_error_price(bi);
        else
            abs_error = market_price - model_price;
            errors(ai) = errors(ai) + sum(abs_error(bi).^2,2);
        end
    end
end
% If no data in relevant date range, do nothing
if (~exist('errors','var'))
    cds_new = cds;
    rmse = 0;
    return;
end

% Calculate RMSE and return updated CDS object
cds_new = cds;
if (~isfield(cds_new, 'rmse'))
    cds_new.rmse = zeros(length(cds.dates{1}), 1);
end
cds_new.rmse(used_range_5yr) = sqrt(errors ./ num_prices);
if (~isempty(used_range_5yr))
    rmse = sqrt(sum(errors ./ num_prices) / length(used_range));
else
    warning(['RMSE for ' cds.ticker ' could not be calculated, since date range is empty']);
    rmse = 1e3;
end
